<?php
/**
 * Binance Pay (Note Verification) Gateway Module for WHMCS
 *
 * This module allows clients to pay invoices using Binance Pay.  A unique
 * payment note is generated for each invoice and the module polls the
 * Binance API to verify incoming transactions.  It supports manual USD to
 * BDT conversion and optional gateway fees.  When verifying payments,
 * amounts are always compared in USD, but invoices in other currencies
 * (such as BDT) are automatically converted based upon the configured
 * conversion rate.  This ensures invoices are marked fully paid, rather
 * than recorded as partial payments, regardless of the client's chosen
 * currency.
 *
 * Place this file into your WHMCS installation under
 * modules/gateways/binance_note.php.  Two additional scripts
 * (binance_note_pay.php and binance_note_verify.php) are required for
 * processing the payment page and verifying transactions.  See those
 * files for further details.
 */

if (!defined('WHMCS')) {
    die('This file cannot be accessed directly');
}

/**
 * Module configuration
 *
 * Defines all settings visible in the WHMCS admin panel.  Administrators
 * should enter their Binance API credentials here and set the USD→BDT
 * conversion rate to match their local currency.  The fee settings allow
 * an optional fixed or percentage gateway fee to be added to each invoice.
 *
 * @return array
 */
function binance_note_config()
{
    return [
        'FriendlyName' => [
            'Type'  => 'System',
            'Value' => 'Binance Pay (Note Verification)',
        ],
        'apiKey' => [
            'FriendlyName' => 'Binance API Key',
            'Type'         => 'text',
            'Size'         => '50',
        ],
        'secretKey' => [
            'FriendlyName' => 'Binance Secret Key',
            'Type'         => 'text',
            'Size'         => '80',
        ],
        'binanceUid' => [
            'FriendlyName' => 'Binance Pay UID',
            'Type'         => 'text',
            'Size'         => '40',
        ],
        'qrImageUrl' => [
            'FriendlyName' => 'QR Code Image URL',
            'Type'         => 'text',
            'Size'         => '255',
            'Description'  => '<a href="/modules/gateways/binancenote/uploader.php">Upload QR</a> , Optional QR image URL',
        ],
        'baseUrl' => [
            'FriendlyName' => 'Binance API Base URL',
            'Type'         => 'text',
            'Size'         => '80',
            'Default'      => 'https://api.binance.com',
        ],
        'usdToBdtRate' => [
            'FriendlyName' => 'USD → BDT Rate',
            'Type'         => 'text',
            'Size'         => '10',
            'Default'      => '110',
            'Description'  => 'Manual conversion rate (1 USD = ? BDT)',
        ],
        'feeType' => [
            'FriendlyName' => 'Gateway Fee Type',
            'Type'         => 'dropdown',
            'Options'      => 'None,Fixed,Percentage',
            'Description'  => 'Select if you want to add a gateway fee',
        ],
        'feeValue' => [
            'FriendlyName' => 'Gateway Fee Value',
            'Type'         => 'text',
            'Size'         => '10',
            'Default'      => '0',
            'Description'  => 'Fixed fee amount or percentage (e.g., 2 = 2%)',
        ],
    ];
}

/**
 * Convert between currencies using a manual USD→BDT rate.
 *
 * This function currently supports conversion between USD and BDT.  You can
 * extend it to support additional currencies if required by adding more
 * conditions or integrating with WHMCS's convertCurrency() helper.  The
 * rate is read from the gateway configuration to allow administrators
 * fine‑control over the conversion used by this module.
 *
 * @param float  $amount    The numeric amount to convert
 * @param string $from      The source currency code (e.g. USD or BDT)
 * @param string $to        The destination currency code
 * @param array  $params    Module configuration parameters
 *
 * @return float The converted amount rounded to 2 decimals
 */
function binance_convert_currency($amount, $from, $to, $params)
{
    $rate = isset($params['usdToBdtRate']) ? (float) $params['usdToBdtRate'] : 110;
    if ($rate <= 0) {
        $rate = 110;
    }
    $from = strtoupper($from);
    $to   = strtoupper($to);
    // 1 USD = {rate} BDT
    if ($from === 'USD' && $to === 'BDT') {
        return round($amount * $rate, 2);
    } elseif ($from === 'BDT' && $to === 'USD') {
        return round($amount / $rate, 2);
    }
    // If converting the same currency or unsupported pair, return original
    return round($amount, 2);
}

/**
 * Apply a gateway fee to an amount.
 *
 * Depending on the configuration, a fixed or percentage fee can be added to
 * the invoice total.  The fee value is applied before any currency
 * conversion takes place.
 *
 * @param float  $amount   The base amount before fees
 * @param string $currency The currency code of the amount
 * @param array  $params   Module configuration parameters
 *
 * @return float The amount including any gateway fee
 */
function binance_apply_gateway_fee($amount, $currency, $params)
{
    $feeType  = $params['feeType']  ?? 'None';
    $feeValue = isset($params['feeValue']) ? (float) $params['feeValue'] : 0.0;
    if ($feeType === 'Fixed' && $feeValue > 0) {
        $amount += $feeValue;
    } elseif ($feeType === 'Percentage' && $feeValue > 0) {
        $amount += $amount * ($feeValue / 100.0);
    }
    return round($amount, 2);
}

/**
 * Generate a payment link for an invoice.
 *
 * When WHMCS renders the invoice view, it will call this function to obtain
 * the HTML link or button that directs the client to the payment page.  The
 * link includes the invoice ID and the amount to be paid in USD.  The
 * amount shown on the button includes any gateway fee and displays both
 * USD and the client’s currency for clarity.
 *
 * @param array $params Standard WHMCS gateway variables
 *
 * @return string HTML output for the invoice payment button
 */
function binance_note_link($params)
{
    $invoiceId = $params['invoiceid'] ?? 0;
    $amount    = isset($params['amount']) ? (float) $params['amount'] : 0.0;
    $currency  = strtoupper($params['currency'] ?? 'USD');

    // Apply gateway fee on the invoice amount (in invoice currency)
    $amountWithFee = binance_apply_gateway_fee($amount, $currency, $params);

    // Convert the amount (including fee) to USD for Binance
    $usdAmount = binance_convert_currency($amountWithFee, $currency, 'USD', $params);

    // Prepare a display amount in the invoice currency for clarity
    $displayAmount = number_format($amountWithFee, 2, '.', '');
    $displayUSD    = number_format($usdAmount, 2, '.', '');

    // Build the URL to the payment page
    $url = 'modules/gateways/binancenote/binance_note_pay.php?invoiceid=' . $invoiceId;

    // Display both currencies if the invoice currency is not USD
    $buttonLabel = '';
    if ($currency === 'USD') {
        $buttonLabel = 'Pay Now with Binance (USD ' . $displayUSD . ')';
    } else {
        $buttonLabel = 'Pay Now with Binance (USD ' . $displayUSD . ' / ' . $currency . ' ' . $displayAmount . ')';
    }

    return '<a href="' . $url . '" style="padding:10px 15px; background:#007bff; color:#fff; text-decoration:none; border-radius:5px;">' . $buttonLabel . '</a>';
}
