<?php
session_start();

// Configuration
$uploadDir = __DIR__ . '/qrcodes/';
$allowedTypes = ['image/png', 'image/jpeg', 'image/jpg'];
$maxFileSize = 5 * 1024 * 1024; // 5MB
$baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://$_SERVER[HTTP_HOST]";

// Create folder if not exists with proper permissions
if (!is_dir($uploadDir)) {
    if (!mkdir($uploadDir, 0755, true)) {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Failed to create upload directory']);
        exit;
    }
}

// Security: Prevent directory traversal
if (isset($_POST['file']) && isset($_POST['action']) && $_POST['action'] === 'delete') {
    $filename = basename($_POST['file']);
    $filePath = $uploadDir . $filename;
    
    if (file_exists($filePath)) {
        unlink($filePath);
        echo json_encode(['success' => true, 'message' => 'File deleted successfully']);
    } else {
        echo json_encode(['success' => false, 'message' => 'File not found']);
    }
    exit;
}

// Handle AJAX upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['qrfile'])) {
    header('Content-Type: application/json');
    
    // Check for upload errors
    if ($_FILES['qrfile']['error'] !== UPLOAD_ERR_OK) {
        $errorMessages = [
            UPLOAD_ERR_INI_SIZE => 'File exceeds upload_max_filesize directive',
            UPLOAD_ERR_FORM_SIZE => 'File exceeds MAX_FILE_SIZE directive',
            UPLOAD_ERR_PARTIAL => 'File was only partially uploaded',
            UPLOAD_ERR_NO_FILE => 'No file was uploaded',
            UPLOAD_ERR_NO_TMP_DIR => 'Missing temporary folder',
            UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk',
            UPLOAD_ERR_EXTENSION => 'A PHP extension stopped the file upload'
        ];
        
        $error = $_FILES['qrfile']['error'];
        $message = $errorMessages[$error] ?? 'Unknown upload error';
        
        echo json_encode(['success' => false, 'message' => $message]);
        exit;
    }
    
    // Validate file type
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $fileType = finfo_file($finfo, $_FILES['qrfile']['tmp_name']);
    finfo_close($finfo);
    
    if (!in_array($fileType, $allowedTypes)) {
        echo json_encode(['success' => false, 'message' => 'Only PNG and JPG images are allowed']);
        exit;
    }
    
    // Validate file size
    if ($_FILES['qrfile']['size'] > $maxFileSize) {
        echo json_encode(['success' => false, 'message' => 'File size must be less than 5MB']);
        exit;
    }
    
    // Generate a unique filename while keeping the original extension
    $ext = strtolower(pathinfo($_FILES['qrfile']['name'], PATHINFO_EXTENSION));
    $fileName = 'qrcode_' . time() . '_' . bin2hex(random_bytes(5)) . '.' . $ext;
    $destPath = $uploadDir . $fileName;
    
    // Validate image content
    $imageInfo = getimagesize($_FILES['qrfile']['tmp_name']);
    if (!$imageInfo) {
        echo json_encode(['success' => false, 'message' => 'Uploaded file is not a valid image']);
        exit;
    }
    
    // Move uploaded file
    if (move_uploaded_file($_FILES['qrfile']['tmp_name'], $destPath)) {
        // Store filename in session for management
        $_SESSION['uploaded_qr'] = $fileName;
        
        echo json_encode([
            'success' => true,
            'message' => 'QR code uploaded successfully',
            'file' => 'qrcodes/' . $fileName,
            'filename' => $fileName,
            'fileUrl' => $baseUrl . '/qrcodes/' . $fileName
        ]);
    } else {
        echo json_encode(['success' => false, 'message' => 'Failed to save file']);
    }
    exit;
}

// Handle file listing
if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['action']) && $_GET['action'] === 'list') {
    $files = [];
    if (is_dir($uploadDir)) {
        $fileList = scandir($uploadDir);
        foreach ($fileList as $file) {
            if ($file !== '.' && $file !== '..' && !is_dir($uploadDir . $file)) {
                $files[] = [
                    'name' => $file,
                    'url' => 'qrcodes/' . $file,
                    'fileUrl' => $baseUrl . '/qrcodes/' . $file,
                    'size' => filesize($uploadDir . $file),
                    'date' => filemtime($uploadDir . $file)
                ];
            }
        }
    }
    usort($files, function($a, $b) {
        return $b['date'] - $a['date'];
    });
    
    echo json_encode(['success' => true, 'files' => $files]);
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>QR Code Manager</title>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-color: #4361ee;
            --success-color: #06d6a0;
            --danger-color: #ef476f;
            --warning-color: #ffd166;
            --dark-color: #2b2d42;
            --light-color: #f8f9fa;
        }
        
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            line-height: 1.6;
            color: #333;
            background-color: #f5f7ff;
            padding: 20px;
        }
        
        .container {
            max-width: 1000px;
            margin: 0 auto;
            background: white;
            border-radius: 12px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
            padding: 30px;
        }
        
        h1, h2 {
            color: var(--dark-color);
            margin-bottom: 20px;
        }
        
        h1 {
            text-align: center;
            margin-bottom: 30px;
            padding-bottom: 15px;
            border-bottom: 1px solid #eaeaea;
        }
        
        .upload-section, .gallery-section {
            margin-bottom: 40px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .file-input {
            position: relative;
            display: inline-block;
            width: 100%;
        }
        
        .file-input input[type="file"] {
            position: absolute;
            left: 0;
            top: 0;
            opacity: 0;
            width: 100%;
            height: 100%;
            cursor: pointer;
        }
        
        .file-input-label {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            padding: 40px 20px;
            border: 2px dashed #ccc;
            border-radius: 8px;
            background-color: #fafafa;
            text-align: center;
            transition: all 0.3s;
        }
        
        .file-input-label:hover {
            border-color: var(--primary-color);
            background-color: #f0f4ff;
        }
        
        .file-input-label i {
            font-size: 48px;
            margin-bottom: 15px;
            color: #aaa;
        }
        
        .file-input-label.dragover {
            border-color: var(--primary-color);
            background-color: #e6eeff;
        }
        
        .btn {
            display: inline-block;
            padding: 10px 20px;
            background-color: var(--primary-color);
            color: white;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 500;
            transition: all 0.3s;
            text-decoration: none;
        }
        
        .btn:hover {
            background-color: #3a56d4;
            transform: translateY(-2px);
        }
        
        .btn:disabled {
            background-color: #ccc;
            cursor: not-allowed;
        }
        
        .btn-sm {
            padding: 6px 12px;
            font-size: 12px;
        }
        
        .btn-success {
            background-color: var(--success-color);
        }
        
        .btn-success:hover {
            background-color: #05c290;
        }
        
        .btn-danger {
            background-color: var(--danger-color);
        }
        
        .btn-danger:hover {
            background-color: #e5366a;
        }
        
        .progress-container {
            margin-top: 20px;
            display: none;
        }
        
        .progress-bar {
            height: 10px;
            background-color: #eaeaea;
            border-radius: 5px;
            overflow: hidden;
        }
        
        .progress-bar-fill {
            height: 100%;
            background-color: var(--primary-color);
            width: 0%;
            transition: width 0.3s;
        }
        
        .alert {
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
            display: none;
        }
        
        .alert-success {
            background-color: #e6f7f2;
            color: var(--success-color);
            border: 1px solid #b6efdf;
        }
        
        .alert-error {
            background-color: #fde8ec;
            color: var(--danger-color);
            border: 1px solid #f9c6d1;
        }
        
        .gallery {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }
        
        .qr-item {
            background: white;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 3px 10px rgba(0, 0, 0, 0.08);
            transition: transform 0.3s;
        }
        
        .qr-item:hover {
            transform: translateY(-5px);
        }
        
        .qr-image {
            width: 100%;
            height: 180px;
            object-fit: cover;
            border-bottom: 1px solid #eaeaea;
        }
        
        .qr-details {
            padding: 15px;
        }
        
        .qr-name {
            font-weight: 500;
            margin-bottom: 5px;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        
        .qr-meta {
            font-size: 12px;
            color: #777;
            margin-bottom: 10px;
        }
        
        .qr-actions {
            display: flex;
            justify-content: space-between;
            flex-wrap: wrap;
            gap: 5px;
        }
        
        .link-section {
            margin-top: 15px;
            padding-top: 15px;
            border-top: 1px dashed #eee;
        }
        
        .link-container {
            display: flex;
            margin-top: 8px;
        }
        
        .link-input {
            flex-grow: 1;
            padding: 8px 12px;
            border: 1px solid #ddd;
            border-radius: 4px 0 0 4px;
            font-size: 12px;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        
        .copy-btn {
            border-radius: 0 4px 4px 0;
            border-left: none;
        }
        
        .empty-state {
            text-align: center;
            padding: 40px 20px;
            color: #999;
            grid-column: 1 / -1;
        }
        
        .empty-state i {
            font-size: 64px;
            margin-bottom: 15px;
            display: block;
        }
        
        @media (max-width: 768px) {
            .container {
                padding: 15px;
            }
            
            .gallery {
                grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
            }
            
            .qr-actions {
                flex-direction: column;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>QR Code Manager</h1>
        
        <section class="upload-section">
            <h2>Upload New QR Code</h2>
            
            <div class="alert alert-success" id="successAlert"></div>
            <div class="alert alert-error" id="errorAlert"></div>
            
            <form id="qrUploader" enctype="multipart/form-data" method="post">
                <div class="form-group">
                    <div class="file-input">
                        <input type="file" name="qrfile" id="qrfile" accept=".png,.jpg,.jpeg" required>
                        <label for="qrfile" class="file-input-label" id="fileInputLabel">
                            <i class="fas fa-cloud-upload-alt"></i>
                            <span>Click to select or drag and drop a QR code image</span>
                            <small>(PNG or JPG, max 5MB)</small>
                        </label>
                    </div>
                </div>
                
                <div class="form-group">
                    <button type="submit" class="btn" id="uploadBtn">Upload QR Code</button>
                </div>
            </form>
            
            <div class="progress-container" id="progressContainer">
                <div class="progress-bar">
                    <div class="progress-bar-fill" id="progressBar"></div>
                </div>
                <div id="progressText">0%</div>
            </div>
        </section>
        
        <section class="gallery-section">
            <h2>Uploaded QR Codes</h2>
            <div class="gallery" id="qrGallery">
                <div class="empty-state">
                    <i class="fas fa-qrcode"></i>
                    <p>No QR codes uploaded yet</p>
                </div>
            </div>
        </section>
    </div>

    <script>
    $(document).ready(function() {
        // Load existing QR codes
        loadQRCodes();
        
        // Drag and drop handling
        const fileInputLabel = $('#fileInputLabel');
        const fileInput = $('#qrfile');
        
        fileInputLabel.on('dragover', function(e) {
            e.preventDefault();
            e.stopPropagation();
            $(this).addClass('dragover');
        });
        
        fileInputLabel.on('dragleave', function(e) {
            e.preventDefault();
            e.stopPropagation();
            $(this).removeClass('dragover');
        });
        
        fileInputLabel.on('drop', function(e) {
            e.preventDefault();
            e.stopPropagation();
            $(this).removeClass('dragover');
            
            const files = e.originalEvent.dataTransfer.files;
            if (files.length > 0) {
                fileInput[0].files = files;
                updateFileName(files[0].name);
            }
        });
        
        fileInput.on('change', function() {
            if (this.files.length > 0) {
                updateFileName(this.files[0].name);
            }
        });
        
        function updateFileName(name) {
            fileInputLabel.html(`<i class="fas fa-file-image"></i><span>Selected file: <strong>${name}</strong></span><small>Click or drag to change</small>`);
        }
        
        // Form submission with AJAX
        $("#qrUploader").on("submit", function(e) {
            e.preventDefault();
            
            const file = fileInput[0].files[0];
            if (!file) {
                showError('Please select a file to upload');
                return;
            }
            
            const formData = new FormData(this);
            const uploadBtn = $('#uploadBtn');
            
            // Show progress
            $('#progressContainer').show();
            uploadBtn.prop('disabled', true);
            
            $.ajax({
                url: "uploader.php",
                type: "POST",
                data: formData,
                contentType: false,
                processData: false,
                xhr: function() {
                    const xhr = new window.XMLHttpRequest();
                    xhr.upload.addEventListener("progress", function(evt) {
                        if (evt.lengthComputable) {
                            const percentComplete = (evt.loaded / evt.total) * 100;
                            $('#progressBar').css('width', percentComplete + '%');
                            $('#progressText').text(Math.round(percentComplete) + '%');
                        }
                    }, false);
                    return xhr;
                },
                success: function(response) {
                    try {
                        const res = typeof response === 'string' ? JSON.parse(response) : response;
                        
                        if (res.success) {
                            showSuccess(res.message);
                            loadQRCodes(); // Refresh the gallery
                            $("#qrUploader")[0].reset();
                            fileInputLabel.html('<i class="fas fa-cloud-upload-alt"></i><span>Click to select or drag and drop a QR code image</span><small>(PNG or JPG, max 5MB)</small>');
                        } else {
                            showError(res.message);
                        }
                    } catch (e) {
                        showError('Unexpected error: ' + e.message);
                    }
                },
                error: function(xhr, status, error) {
                    showError('Upload failed: ' + error);
                },
                complete: function() {
                    $('#progressContainer').hide();
                    $('#progressBar').css('width', '0%');
                    $('#progressText').text('0%');
                    uploadBtn.prop('disabled', false);
                }
            });
        });
        
        // Load QR codes from server
        function loadQRCodes() {
            $.get('uploader.php?action=list', function(response) {
                try {
                    const res = typeof response === 'string' ? JSON.parse(response) : response;
                    
                    if (res.success && res.files.length > 0) {
                        let galleryHtml = '';
                        
                        res.files.forEach(function(file) {
                            const date = new Date(file.date * 1000).toLocaleString();
                            const size = formatFileSize(file.size);
                            
                            galleryHtml += `
                                <div class="qr-item">
                                    <img src="${file.url}?t=${file.date}" alt="${file.name}" class="qr-image">
                                    <div class="qr-details">
                                        <div class="qr-name" title="${file.name}">${file.name}</div>
                                        <div class="qr-meta">
                                            <div>${size}</div>
                                            <div>${date}</div>
                                        </div>
                                        <div class="link-section">
                                            <div><strong>Direct Link:</strong></div>
                                            <div class="link-container">
                                                <input type="text" class="link-input" value="${file.fileUrl}" readonly>
                                                <button class="btn btn-sm btn-success copy-btn" data-link="${file.fileUrl}">
                                                    <i class="fas fa-copy"></i>
                                                </button>
                                            </div>
                                        </div>
                                        <div class="qr-actions">
                                            <a href="${file.url}" download="${file.name}" class="btn btn-sm">
                                                <i class="fas fa-download"></i> Download
                                            </a>
                                            <button class="btn btn-sm btn-danger" onclick="deleteQRCode('${file.name}')">
                                                <i class="fas fa-trash"></i> Delete
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            `;
                        });
                        
                        $('#qrGallery').html(galleryHtml);
                        
                        // Add click event to copy buttons
                        $('.copy-btn').on('click', function() {
                            const link = $(this).data('link');
                            copyToClipboard(link);
                        });
                    } else {
                        $('#qrGallery').html(`
                            <div class="empty-state">
                                <i class="fas fa-qrcode"></i>
                                <p>No QR codes uploaded yet</p>
                            </div>
                        `);
                    }
                } catch (e) {
                    console.error('Error parsing response:', e);
                }
            }).fail(function() {
                showError('Failed to load QR codes');
            });
        }
        
        // Show success message
        function showSuccess(message) {
            $('#errorAlert').hide();
            $('#successAlert').text(message).show();
            setTimeout(() => $('#successAlert').fadeOut(), 5000);
        }
        
        // Show error message
        function showError(message) {
            $('#successAlert').hide();
            $('#errorAlert').text(message).show();
            setTimeout(() => $('#errorAlert').fadeOut(), 5000);
        }
        
        // Format file size
        function formatFileSize(bytes) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        }
        
        // Copy to clipboard function
        function copyToClipboard(text) {
            navigator.clipboard.writeText(text).then(function() {
                showSuccess('Link copied to clipboard!');
            }, function() {
                // Fallback for older browsers
                const textArea = document.createElement("textarea");
                textArea.value = text;
                textArea.style.position = "fixed";
                document.body.appendChild(textArea);
                textArea.focus();
                textArea.select();
                
                try {
                    const successful = document.execCommand('copy');
                    if (successful) {
                        showSuccess('Link copied to clipboard!');
                    } else {
                        showError('Failed to copy link');
                    }
                } catch (err) {
                    showError('Failed to copy link: ' + err);
                }
                
                document.body.removeChild(textArea);
            });
        }
    });
    
    // Delete QR code
    function deleteQRCode(filename) {
        if (!confirm('Are you sure you want to delete this QR code?')) {
            return;
        }
        
        $.post('uploader.php', {file: filename, action: 'delete'}, function(response) {
            try {
                const res = typeof response === 'string' ? JSON.parse(response) : response;
                
                if (res.success) {
                    // Reload the gallery
                    $.get('uploader.php?action=list', function(response) {
                        const res = typeof response === 'string' ? JSON.parse(response) : response;
                        
                        if (res.success && res.files.length > 0) {
                            let galleryHtml = '';
                            
                            res.files.forEach(function(file) {
                                const date = new Date(file.date * 1000).toLocaleString();
                                const size = formatFileSize(file.size);
                                
                                galleryHtml += `
                                    <div class="qr-item">
                                        <img src="${file.url}?t=${file.date}" alt="${file.name}" class="qr-image">
                                        <div class="qr-details">
                                            <div class="qr-name" title="${file.name}">${file.name}</div>
                                            <div class="qr-meta">
                                                <div>${size}</div>
                                                <div>${date}</div>
                                            </div>
                                            <div class="link-section">
                                                <div><strong>Direct Link:</strong></div>
                                                <div class="link-container">
                                                    <input type="text" class="link-input" value="${file.fileUrl}" readonly>
                                                    <button class="btn btn-sm btn-success copy-btn" data-link="${file.fileUrl}">
                                                        <i class="fas fa-copy"></i>
                                                    </button>
                                                </div>
                                            </div>
                                            <div class="qr-actions">
                                                <a href="${file.url}" download="${file.name}" class="btn btn-sm">
                                                    <i class="fas fa-download"></i> Download
                                                </a>
                                                <button class="btn btn-sm btn-danger" onclick="deleteQRCode('${file.name}')">
                                                    <i class="fas fa-trash"></i> Delete
                                                </button>
                                            </div>
                                        </div>
                                    </div>
                                `;
                            });
                            
                            $('#qrGallery').html(galleryHtml);
                            
                            // Reattach event listeners to new copy buttons
                            $('.copy-btn').on('click', function() {
                                const link = $(this).data('link');
                                copyToClipboard(link);
                            });
                        } else {
                            $('#qrGallery').html(`
                                <div class="empty-state">
                                    <i class="fas fa-qrcode"></i>
                                    <p>No QR codes uploaded yet</p>
                                </div>
                            `);
                        }
                    });
                } else {
                    alert('Error: ' + res.message);
                }
            } catch (e) {
                alert('Error parsing response: ' + e.message);
            }
        });
    }
    </script>
</body>
</html>